﻿/*  Author:     Ben Hall
 *  File:       GameMap.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    GameMap.cs is used to represent maps in the game, using boolean arrays to determine
 *              where a player is able to move, as well as MapEvent arrays to show where events are
 *              located, so they can be triggered when the player enters that space.
 */

using System;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Ink;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using System.Windows.Media.Imaging;

namespace OnlineGame
{
    public struct Position
    {
        public int xPos;
        public int yPos;
        private const int MAX_X = (GameMap.MAP_WIDTH - 1);
        private const int MAX_Y = (GameMap.MAP_LENGTH - 1);
        private const int MIN_X = 0;
        private const int MIN_Y = 0;

        public Position(int xIn, int yIn)
        {
            this.xPos = xIn;
            this.yPos = yIn;
        }

        public int X
        {
            get
            {
                return xPos;
            }
            set
            {
                int xValue = value;
                if ((xValue < MIN_X) || (xValue > MAX_X))
                {
                    xValue = MIN_X;
                }
                xPos = xValue;
            }
        }

        public int Y
        {
            get
            {
                return yPos;
            }
            set
            {
                int yValue = value;
                if ((yValue < MIN_Y) || (yValue > MAX_Y))
                {
                    yValue = MIN_Y;
                }
                yPos = yValue;
            }
        }

        public Boolean moveIsInBounds(Position destination)
        {
            if ((destination.X > MAX_X) || (destination.X < MIN_X) || (destination.Y > MAX_Y) || (destination.Y < MIN_Y))
            {
                //goes outside of bounds
                return false;
            }
            else 
            {
                return true;
            }
        }
    }

    public class GameMap
    {
        public const int MAP_WIDTH = 15;
        public const int MAP_LENGTH = 10;

        private String mapID;
        private bool[,] mapMoveGrid = new bool[MAP_WIDTH, MAP_LENGTH];
        private MapEvent[,] mapEventGrid = new MapEvent[MAP_WIDTH, MAP_LENGTH];
        private BitmapImage background;

        private Encounter[] encounterList;
        private float encounterRate;



        //Empty GameMap constructor used when initializing the game state
        public GameMap()
        {
            //this.connectedMaps = new GameMap[0];
            for (int i = 0; i < MAP_WIDTH; i++)
            {
                for (int j = 0; j < MAP_LENGTH; j++)
                {
                    mapMoveGrid[i,j] = true;
                    mapEventGrid[i, j] = null;
                }
            }
            this.encounterRate = 0;
            this.encounterList = new Encounter[0];
            this.background = new BitmapImage(new Uri("Images/testblock.jpg", UriKind.Relative));
        }


        public GameMap(BitmapImage backgroundIn, bool[,] moveGridIn, String mapIDIn, float encounterRateIn, Encounter[] encounterListIn)
        {
            this.mapID = mapIDIn;
            this.encounterRate = encounterRateIn;
            this.background = backgroundIn;
            this.encounterList = encounterListIn;
            if ((moveGridIn.GetLength(0) == mapMoveGrid.GetLength(0)) && (moveGridIn.GetLength(1) == (mapMoveGrid.GetLength(1))))
            {
                this.mapMoveGrid = moveGridIn;
            }
            else
            {
                //ERROR WHEN GRID DIMENSIONS DO NOT MATCH
                for (int i = 0; i < MAP_WIDTH; i++)
                {
                    for (int j = 0; j < MAP_LENGTH; j++)
                    {
                        mapMoveGrid[i,j] = true;
                        //mapEventGrid[i,j] 
                    }
                }
            }

            //initialize MapMoveEvent array to null
            for (int i = 0; i < MAP_WIDTH; i++)
            {
                for (int j = 0; j < MAP_LENGTH; j++)
                {
                    //mapConnections[i, j] = null;
                    mapEventGrid[i, j] = null;
                }
            }
        }

        /*  MoveIsValid() takes in a Position and returns true if the value in the mapMoveGrid
         *  is true at that location.
         */
        public Boolean MoveIsValid(OnlineGame.Position destinationPos)
        {
            Boolean validMove = false;
            if (destinationPos.moveIsInBounds(destinationPos) == true)
            {
                validMove = mapMoveGrid[destinationPos.X, destinationPos.Y];
            }
            else
            {
                validMove = false;
            }
            return validMove;
        }

        public String GetMapID()
        {
            return this.mapID;
        }

        public BitmapImage GetBackground()
        {
            return this.background;
        }

        /*  AddMapMoveEvent() adds a given MapMoveEvent to the map that will trigger at the 
         *  given Position
         */
        public void AddMapMoveEvent(MapMoveEvent eventIn, Position locationIn)
        {
            this.mapEventGrid[locationIn.X, locationIn.Y] = eventIn;
        }

        /*  GetMapMoveEventAtLocation() returns the MapMoveEvent located at the given Position
         */
        public MapMoveEvent GetMapMoveEventAtLocation(Position locationIn)
        {
            if (mapEventGrid[locationIn.X, locationIn.Y] is MapMoveEvent)
            {
                return (MapMoveEvent)mapEventGrid[locationIn.X, locationIn.Y];
            }
            else
            {
                return null;
            }
        }

        public void AddNPCEvent(NPCEvent eventIn, Position locationIn)
        {
            this.mapEventGrid[locationIn.X, locationIn.Y] = eventIn;
        }

        public NPCEvent GetNPCEventAtLocation(Position locationIn)
        {
            if (mapEventGrid[locationIn.X, locationIn.Y] is NPCEvent)
            {
                return (NPCEvent)mapEventGrid[locationIn.X, locationIn.Y];
            }
            else
            {
                return null;
            }
        }

        public MapEvent GetMapEventAtLocation(Position locationIn)
        {
            return mapEventGrid[locationIn.X, locationIn.Y];
        }

        public float GetEncounterRate()
        {
            return this.encounterRate;
        }

        public Encounter[] GetEncounterList()
        {
            return this.encounterList;
        }
    }
}
